<?php

/**
 * Debug script for MultiPOS 500 error
 * Run: php test_multipos_debug.php
 */

require __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

echo "=== MultiPOS Module Debug ===\n\n";

// 1. Check module exists
echo "1. Checking module...\n";
$module = \App\Models\Module::where('name', 'MultiPOS')->first();
if ($module) {
    echo "   ✓ Module exists: ID {$module->id}\n";
} else {
    echo "   ✗ Module NOT found\n";
    echo "   Creating module...\n";
    $module = \App\Models\Module::create(['name' => 'MultiPOS']);
    echo "   ✓ Module created: ID {$module->id}\n";
}

// 2. Check if class exists
echo "\n2. Checking classes...\n";
$classes = [
    'Modules\MultiPOS\Entities\PosMachine',
    'Modules\MultiPOS\Services\PosMachineService',
    'Modules\MultiPOS\Livewire\Settings\MultiPosSettings',
    'Modules\MultiPOS\Http\Controllers\SettingsController',
];

foreach ($classes as $class) {
    if (class_exists($class)) {
        echo "   ✓ {$class}\n";
    } else {
        echo "   ✗ {$class} NOT FOUND\n";
    }
}

// 3. Check table exists
echo "\n3. Checking database table...\n";
try {
    $tableExists = \DB::select("SHOW TABLES LIKE 'pos_machines'");
    if (count($tableExists) > 0) {
        echo "   ✓ Table 'pos_machines' exists\n";
    } else {
        echo "   ✗ Table 'pos_machines' NOT found\n";
    }
} catch (\Exception $e) {
    echo "   ✗ Error: " . $e->getMessage() . "\n";
}

// 4. Check route
echo "\n4. Checking routes...\n";
try {
    $routes = \Route::getRoutes();
    $multiposRoutes = [];
    foreach ($routes as $route) {
        if (str_contains($route->uri(), 'multi-pos')) {
            $multiposRoutes[] = $route->uri();
        }
    }
    if (count($multiposRoutes) > 0) {
        echo "   ✓ Routes found:\n";
        foreach ($multiposRoutes as $uri) {
            echo "     - {$uri}\n";
        }
    } else {
        echo "   ✗ No multi-pos routes found\n";
    }
} catch (\Exception $e) {
    echo "   ✗ Error: " . $e->getMessage() . "\n";
}

// 5. Check view exists
echo "\n5. Checking views...\n";
$views = [
    'multipos::settings.index',
    'multipos::livewire.settings.multi-pos-settings',
];

foreach ($views as $view) {
    try {
        if (\View::exists($view)) {
            echo "   ✓ {$view}\n";
        } else {
            echo "   ✗ {$view} NOT found\n";
        }
    } catch (\Exception $e) {
        echo "   ✗ {$view} - Error: " . $e->getMessage() . "\n";
    }
}

// 6. Test service
echo "\n6. Testing service methods...\n";
try {
    $branch = \App\Models\Branch::first();
    if ($branch) {
        $count = \Modules\MultiPOS\Services\PosMachineService::getMachineCount($branch->id);
        echo "   ✓ PosMachineService::getMachineCount() works: {$count}\n";
    } else {
        echo "   ⚠ No branch found to test service\n";
    }
} catch (\Exception $e) {
    echo "   ✗ Service error: " . $e->getMessage() . "\n";
    echo "   Stack trace:\n";
    echo "   " . $e->getTraceAsString() . "\n";
}

// 7. Check Livewire registration
echo "\n7. Checking Livewire registration...\n";
try {
    $livewire = app('livewire');
    echo "   ✓ Livewire is registered\n";
} catch (\Exception $e) {
    echo "   ✗ Livewire error: " . $e->getMessage() . "\n";
}

echo "\n=== Debug Complete ===\n";

