<?php

namespace Modules\MultiPOS\Livewire\Settings;

use Livewire\Component;
use Modules\MultiPOS\Entities\PosMachine;
use Modules\MultiPOS\Services\PosMachineService;
use Jantinnerezo\LivewireAlert\LivewireAlert;

class MultiPosSettings extends Component
{
    use LivewireAlert;

    public $machines = [];
    public $currentBranchId;

    protected $listeners = ['refreshMachines' => '$refresh'];

    public function mount()
    {
        $this->currentBranchId = branch()->id ?? null;
        $this->loadMachines();
    }

    public function loadMachines()
    {
        if (!$this->currentBranchId) {
            $this->machines = [];
            return;
        }

        $this->machines = PosMachine::with(['registeredBy', 'approvedBy', 'declinedBy'])
            ->where('branch_id', $this->currentBranchId)
            ->orderBy('registered_at', 'desc')
            ->get()
            ->toArray();
    }

    public function approve($machineId)
    {
        $machine = PosMachine::find($machineId);
        
        if (!$machine || $machine->status !== 'pending') {
            $this->alert('error', __('multipos::messages.actions.can_only_approve_pending'));
            return;
        }

        $approved = PosMachineService::approveMachine($machine, auth()->id());

        if ($approved) {
            $this->alert('success', __('multipos::messages.actions.approved_successfully'));
            $this->loadMachines();
            $this->dispatch('refreshMachines');
        } else {
            $this->alert('error', __('multipos::messages.actions.approval_failed'));
        }
    }

    public function decline($machineId, $reason = null)
    {
        $machine = PosMachine::find($machineId);
        
        if (!$machine || $machine->status !== 'pending') {
            $this->alert('error', __('multipos::messages.actions.can_only_decline_pending'));
            return;
        }

        $declined = PosMachineService::declineMachine($machine, auth()->id(), $reason);

        if ($declined) {
            $this->alert('success', __('multipos::messages.actions.declined_successfully'));
            $this->loadMachines();
            $this->dispatch('refreshMachines');
        } else {
            $this->alert('error', __('multipos::messages.actions.decline_failed'));
        }
    }

    public function updateAlias($machineId, $alias)
    {
        $machine = PosMachine::find($machineId);
        
        if (!$machine) {
            $this->alert('error', __('multipos::messages.actions.machine_not_found'));
            return;
        }

        $machine->update(['alias' => $alias]);
        $this->alert('success', __('multipos::messages.actions.updated_successfully'));
        $this->loadMachines();
    }

    public function delete($machineId)
    {
        $machine = PosMachine::find($machineId);
        
        if (!$machine) {
            $this->alert('error', __('multipos::messages.actions.machine_not_found'));
            return;
        }

        $machine->delete();
        $this->alert('success', __('multipos::messages.actions.deleted_successfully'));
        $this->loadMachines();
    }

    public function render()
    {
        $this->loadMachines();
        
        $machineCount = $this->currentBranchId ? PosMachineService::getMachineCount($this->currentBranchId) : 0;
        $remainingSlots = $this->currentBranchId ? PosMachineService::getRemainingSlots($this->currentBranchId) : null;

        return view('multipos::livewire.settings.multi-pos-settings', [
            'machineCount' => $machineCount,
            'remainingSlots' => $remainingSlots,
        ]);
    }
}

