<?php

namespace Modules\MultiPOS\Http\Controllers;

use App\Http\Controllers\Controller;
use Modules\MultiPOS\Entities\PosMachine;
use Modules\MultiPOS\Services\PosMachineService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SettingsController extends Controller
{
    /**
     * Display the settings page with machines list
     */
    public function index()
    {
        return view('multipos::settings.index');
    }

    /**
     * Approve a POS machine
     */
    public function approve(Request $request, PosMachine $machine)
    {
        if ($machine->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => __('multipos::messages.actions.can_only_approve_pending'),
            ], 422);
        }

        $approved = PosMachineService::approveMachine($machine, auth()->id());

        if ($approved) {
            return response()->json([
                'success' => true,
                'message' => __('multipos::messages.actions.approved_successfully'),
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => __('multipos::messages.actions.approval_failed'),
        ], 500);
    }

    /**
     * Decline a POS machine
     */
    public function decline(Request $request, PosMachine $machine)
    {
        if ($machine->status !== 'pending') {
            return response()->json([
                'success' => false,
                'message' => __('multipos::messages.actions.can_only_decline_pending'),
            ], 422);
        }

        $validator = Validator::make($request->all(), [
            'reason' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $declined = PosMachineService::declineMachine(
            $machine,
            auth()->id(),
            $request->input('reason')
        );

        if ($declined) {
            return response()->json([
                'success' => true,
                'message' => __('multipos::messages.actions.declined_successfully'),
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => __('multipos::messages.actions.decline_failed'),
        ], 500);
    }

    /**
     * Update a POS machine (alias)
     */
    public function update(Request $request, PosMachine $machine)
    {
        $validator = Validator::make($request->all(), [
            'alias' => 'nullable|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $machine->update($request->only(['alias']));

        return response()->json([
            'success' => true,
            'message' => __('multipos::messages.actions.updated_successfully'),
        ]);
    }

    /**
     * Delete a POS machine
     */
    public function destroy(PosMachine $machine)
    {
        $machine->delete();

        return response()->json([
            'success' => true,
            'message' => __('multipos::messages.actions.deleted_successfully'),
        ]);
    }
}

